/*
 * Copyright (c) 1999-2001 Lutris Technologies, Inc. All Rights
 * Reserved.
 * 
 * This source code file is distributed by Lutris Technologies, Inc. for
 * use only by licensed users of product(s) that include this source
 * file. Use of this source file or the software that uses it is covered
 * by the terms and conditions of the Lutris Enhydra Development License
 * Agreement included with this product.
 * 
 * This Software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF
 * ANY KIND, either express or implied. See the License for the specific terms
 * governing rights and limitations under the License.
 * 
 * Contributor(s):
 * 
 * $Id: Xml.java,v 1.1.1.1 2001/01/05 05:11:17 sese0235 Exp $
 */

package de.kxml;

import java.io.*;
import java.util.*;

/**
 * A class containing several static xml methods, mainly for escaping
 * special characters like angle brakets and quotes. This class
 * contains also some (partially shared) constants for the parser and
 * kDOM.
 */
public class Xml {
    public static final String NO_NAMESPACE = "";

    /**
     * Integer constant for comments
     */
    public static final int    COMMENT = '-';

    /**
     * Integer constant for doctype
     */
    public static final int    DOCTYPE = '!';

    /**
     * Integer constant for elements
     */
    public static final int    ELEMENT = '<';

    /**
     * Integer constant returned by ParseEvent.getType if the end of
     * the document has been reached
     */
    public static final int    END_DOCUMENT = '.';

    /**
     * Integer constant assigned to an EndTag parse event
     */
    public static final int    END_TAG = '/';

    /**
     * Integer constant assigned to a processing instruction
     */
    public static final int    PROCESSING_INSTRUCTION = '?';

    /**
     * Integer constant assigned to StartTag parse event
     */
    public static final int    START_TAG = '<';

    /**
     * Integer constant assigned to a text node
     */
    public static final int    TEXT = 't';

    /**
     * minimum escaping, quotes are not escaped
     */
    public static final int    ENCODE_MIN = 0;

    /**
     * forces escaping of quotes
     */
    public static final int    ENCODE_QUOT = 1;

    /**
     * forces escaping of all character coded greater than 127
     */
    public static int	       ENCODE_128 = 2;

    /**
     * Constant identifying wap extension events
     */
    public static final int    WAP_EXTENSION = 'w';

    /**
     * unescaped an xml encoded string
     */
    public static String decode(String cooked) {
	StringBuffer result = new StringBuffer();
	int	     i0 = 0;

	while (true) {
	    int i1 = cooked.indexOf('&', i0);

	    if (i1 == -1) {
		break;
	    } 

	    result.append(cooked.substring(i0, i1));

	    int i2 = cooked.indexOf(';', i1);

	    if (i2 == -1) {
		result.append("&amp;");

		i0 = i1 + 1;

		break;
	    } 

	    String decode = cooked.substring(i1 + 1, i2);

	    if (decode.startsWith("#x")) {
		result.append((char) Integer.parseInt(decode.substring(2), 
						      16));
	    } else if (decode.startsWith("#")) {
		result.append((char) Integer.parseInt(decode.substring(1)));
	    } else if (decode.equals("lt")) {
		result.append('<');
	    } else if (decode.equals("gt")) {
		result.append('>');
	    } else if (decode.equals("quot")) {
		result.append('"');
	    } else if (decode.equals("amp")) {
		result.append('&');
	    } else if (decode.equals("apos")) {
		result.append('\'');
	    } else {
		throw new RuntimeException("illegal character entity: &" 
					   + decode + ";");
	    }

	    i0 = i2 + 1;
	} 

	result.append(cooked.substring(i0));

	return result.toString();
    } 

    /**
     * convenience method for encode (String raw, ENCODE_MIN)
     */
    public static String encode(String raw) {
	return encode(raw, ENCODE_MIN);
    } 

    /*
     * encodes an attribute with the given name and value.  A single
     * space is inserted before the name of the attribute
     * public static String encodeAttr (String name, String value) {
     * return " "+name+"=\"" + encode (value, ENCODE_QUOT) + "\"";
     * }
     */

    /**
     * encodes a string escaping less than etc.
     */
    public static String encode(String raw, int flags) {
	int	     len = raw.length();
	StringBuffer cooked = new StringBuffer(raw.length());

	for (int i = 0; i < len; i++) {
	    char c = raw.charAt(i);

	    switch (c) {

	    case '<':
		cooked.append("&lt;");

		break;

	    case '>':
		cooked.append("&gt;");

		break;

	    case '&':
		cooked.append("&amp;");

		break;

	    case '"': {
		if ((flags & ENCODE_QUOT) != 0) {
		    cooked.append("&quot;");
		} else {
		    cooked.append('"');
		}
	    } 

		break;

	    default:
		if (c >= 128 && ((flags & ENCODE_128) != 0)) {
		    cooked.append("&#" + ((int) c) + ";");
		} else {
		    cooked.append(c);
		}
	    }
	} 

	return cooked.toString();
    } 

    /**
     * quotes a string by adding quote chars at the beginning
     * and the end and escaping all quotes and XML special
     * chars contained in the string
     */
    public String quote(String q) {
	return "\"" + encode(q, ENCODE_QUOT) + "\"";
    } 

}

